using System;
using System.Runtime.InteropServices;
using System.Text;
using DynaPDF;

namespace text_search
{
   public class CTextSearch
   {
      private const double MAX_LINE_ERROR = 4.0; // This must be the square of the allowed error (2 * 2 in this case).

      protected struct TGState
      {
         public IntPtr    ActiveFont;
         public float     CharSpacing;
         public double    FontSize;
         public TFontType FontType;
         public TCTM      Matrix;
         public float     SpaceWidth;
         public TDrawMode TextDrawMode;
         public float     TextScale;
         public float     WordSpacing;
      }
      protected class CStack
      {
         public bool Restore(ref TGState F)
         {
            if (m_Count > 0)
            {
               --m_Count;
               F = m_Items[m_Count];
               return true;
            }
            return false;
         }
         public int Save(ref TGState F)
         {
            if (m_Count == m_Capacity)
            {
               m_Capacity += 16;
               try
               {
                  TGState[] tmp = new TGState[m_Capacity];
                  if (m_Items != null) m_Items.CopyTo(tmp, 0);
                  m_Items = tmp;
                  tmp = null;
               }catch
               {
                  m_Capacity -= 16;
                  return -1;
               }
            }
            m_Items[m_Count] = F;
            ++m_Count;
            return 0;
         }
         private uint      m_Capacity;
         private uint      m_Count;
         private TGState[] m_Items;
      }

      internal CTextSearch(CPDF PDFInst)
      {
         m_GState.ActiveFont   = IntPtr.Zero;
         m_GState.CharSpacing  = 0.0f;
         m_GState.FontSize     = 1.0;
         m_GState.FontType     = TFontType.ftType1;
         m_GState.Matrix.a     = 1.0;
         m_GState.Matrix.b     = 0.0;
         m_GState.Matrix.c     = 0.0;
         m_GState.Matrix.d     = 1.0;
         m_GState.Matrix.x     = 0.0;
         m_GState.Matrix.y     = 0.0;
         m_GState.SpaceWidth   = 0.0f;
         m_GState.TextDrawMode = TDrawMode.dmNormal;
         m_GState.TextScale    = 100.0f;
         m_GState.WordSpacing  = 0.0f;
         m_LastTextDir         = TTextDir.tfNotInitialized;
         m_OutBuf              = new byte[64];
         m_PDF                 = PDFInst;
         m_Stack               = new CStack();
      }

      public int BeginTemplate(TPDFRect BBox, IntPtr Matrix)
      {
         if (SaveGState() < 0) return -1;
         Reset();
         if (!IntPtr.Zero.Equals(Matrix))
         {
            TCTM m = (TCTM)Marshal.PtrToStructure(Matrix, typeof(TCTM));
            m_GState.Matrix = MulMatrix(m_GState.Matrix, m);
         }
         return 0;
      }

      private double CalcDistance(double x1, double y1, double x2, double y2)
      {
         double dx = x2-x1;
         double dy = y2-y1;
         return Math.Sqrt(dx*dx + dy*dy);
      }

      private bool Compare()
      {
         int i = 0;
         // m_OutBuf contains an Unicode string. TranslateRawCode() uses a byte array because
         // we found no other way to marshal the string.
         String txt = System.Text.UnicodeEncoding.Unicode.GetString(m_OutBuf, 0, m_OutLen << 1);
         while (i < m_OutLen)
         {
            if (m_SearchText[m_SearchPos] != txt[i++])
            {
               m_HavePos   = false;
               m_SearchPos = 0;
               return false;
            }
            ++m_SearchPos;
            if (m_SearchPos == m_SearchText.Length)
            {
               m_SearchPos = 0;
               return (i == m_OutLen);
            }
         }
         return true;
      }

      private bool DrawRect(TCTM Matrix, double EndX)
      {
         // Note that the start and end coordinate can use different transformation matrices
         double x2 = EndX;
         double y2 = 0.0;
         double x3 = EndX;
         double y3 = m_GState.FontSize;
         Transform(Matrix, ref x2, ref y2);
         Transform(Matrix, ref x3, ref y3);
         return DrawRectEx(x2, y2, x3, y3);
      }

      private bool DrawRectEx(double x2, double y2, double x3, double y3)
      {
         m_PDF.MoveTo(m_x1, m_y1);
         m_PDF.LineTo(x2, y2);
         m_PDF.LineTo(x3, y3);
         m_PDF.LineTo(m_x4, m_y4);
         m_HavePos = false;
         ++m_SelCount;
         return m_PDF.ClosePath(DynaPDF.TPathFillMode.fmFill);
      }

      public void EndTemplate()
      {
         RestoreGState();
      }

      public int GetSelCount()
      {
         return m_SelCount;
      }

      public void Init()
      {
         InitGState();
         Reset();
         m_SelCount = 0;
      }

      private void InitGState()
      {
         while (RestoreGState());
         m_GState.ActiveFont   = IntPtr.Zero;
         m_GState.CharSpacing  = 0.0f;
         m_GState.FontSize     = 1.0;
         m_GState.Matrix.a     = 1.0;
         m_GState.Matrix.b     = 0.0;
         m_GState.Matrix.c     = 0.0;
         m_GState.Matrix.d     = 1.0;
         m_GState.Matrix.x     = 0.0;
         m_GState.Matrix.y     = 0.0;
         m_GState.SpaceWidth   = 0.0f;
         m_GState.TextDrawMode = TDrawMode.dmNormal;
         m_GState.TextScale    = 100.0f;
         m_GState.WordSpacing  = 0.0f;
         m_LastTextDir         = TTextDir.tfNotInitialized;
      }

      private bool IsPointOnLine(double x, double y, double x0, double y0, double x1, double y1)
      {
         double dx, dy, di;
         x -= x0;
         y -= y0;
         dx = x1 - x0;
         dy = y1 - y0;
         di = (x*dx + y*dy) / (dx*dx + dy*dy);
         di = (di < 0.0) ? 0.0 : (di > 1.0) ? 1.0 : di;
         dx = x - di * dx;
         dy = y - di * dy;
         di = dx*dx + dy*dy;
         return (di < MAX_LINE_ERROR);
      }

      private bool MarkSubString(ref double x, TCTM Matrix, TTextRecordA Source)
      {
         int i = 0, decoded = 0;
         float spaceWidth2 = -m_GState.SpaceWidth * 6.0f;
         double width = 0.0;
         int max = Source.Length;
         long src = (long)Source.Text;
         if (Source.Advance < -m_GState.SpaceWidth)
         {
            // If the distance is too large then we assume that no space was emulated at this position.
            if (Source.Advance > spaceWidth2 && m_SearchText[m_SearchPos] == 32)
            {
               if (!m_HavePos)
               {
                  SetStartCoord(Matrix, x);
                  ++m_SearchPos;
                  if (m_SearchPos == m_SearchText.Length)
                  {
                     if (!DrawRect(Matrix, x - Source.Advance)) return false; 
                     Reset();
                  }
               }else if (m_SearchPos == m_SearchText.Length)
               {
                  if (!DrawRect(Matrix, 0.0)) return false;
                  Reset();
               }else
                  ++m_SearchPos;
            }else
               Reset();
         }
         x -= Source.Advance;
         while (i < max)
         {
            i += m_PDF.TranslateRawCode(m_GState.ActiveFont, new IntPtr(src + i), max - i, ref width, m_OutBuf, ref m_OutLen, ref decoded, m_GState.CharSpacing, m_GState.WordSpacing, m_GState.TextScale);
            // We skip the text record if the text cannot be converted to Unicode. The return value must be true,
            // otherwise we would break processing.
            if (decoded == 0) return true;
            // m_OutLen is always greater zero if decoded is true!
            if (Compare())
            {
               if (!m_HavePos)
               {
                  SetStartCoord(Matrix, x);
               }
               x += width;
               if (m_SearchPos == 0)
               {
                  if (!DrawRect(Matrix, x - m_GState.CharSpacing)) return false;
               }
            }else
               x += width;
         }
         return true;
      }

      public int MarkText(TCTM Matrix, TTextRecordA[] Source, int Count, double Width)
      {
         try
         {
            /*
               Note that we write rectangles to the page while we parsed it. This is critical because the parser
               doesn't notice when a fatal error occurs, e.g. out of memory. We must make sure that processing
               breaks immediately in such a case. To archive this the return value of ClosePath() is checked each
               time a rectangle is drawn to the page. The function can only fail if a fatal error occurred.
            */
            int i;
            TTextDir textDir;
            double x;
            double x1 = 0.0;
            double y1 = 0.0;
            double x2 = 0.0;
            double y2 = m_GState.FontSize;
            // Transform the text matrix to user space
            TCTM m = MulMatrix(m_GState.Matrix, Matrix);
            Transform(m, ref x1, ref y1); // Start point of the text record
            Transform(m, ref x2, ref y2); // Second point to determine the text direction
            // Determine the text direction
            if (y1 == y2)
               textDir = (TTextDir)((System.Convert.ToInt32(x1 > x2) + 1) << 1);
            else
               textDir = (TTextDir)System.Convert.ToInt32(y1 > y2);

            if (textDir != m_LastTextDir || !IsPointOnLine(x1, y1, m_LastTextEndX1, m_LastTextEndY1, m_LastTextInfX, m_LastTextInfY))
            {
               // Extend the x-coordinate to an infinite point
               m_LastTextInfX = 1000000.0;
               m_LastTextInfY = 0.0;
               Transform(m, ref m_LastTextInfX, ref m_LastTextInfY);
               Reset();
            }else
            {
               double distance, spaceWidth;
               double x3 = m_GState.SpaceWidth;
               double y3 = 0.0;
               Transform(m, ref x3, ref y3);
               spaceWidth = CalcDistance(x1, y1, x3 ,y3);
               distance   = CalcDistance(m_LastTextEndX1, m_LastTextEndY1, x1, y1);
               if (distance > spaceWidth)
               {
                  // If the distance is too large then we assume that no space was emulated at this position.
                  if (distance < spaceWidth * 6.0 && m_SearchText[m_SearchPos] == 32)
                  {
                     if (!m_HavePos)
                     {
                        // The start coordinate is the end coordinate of the last text record.
                        m_HavePos = true;
                        ++m_SearchPos;
                        if (m_SearchPos == m_SearchText.Length)
                        {
                           m_x1 = m_LastTextEndX1;
                           m_y1 = m_LastTextEndY1;
                           m_x4 = m_LastTextEndX4;
                           m_y4 = m_LastTextEndY4;
                           if (!DrawRectEx(x1, y1, x2, y2)) return -1;
                           Reset();
                        }
                     }else if (m_SearchPos == m_SearchText.Length)
                     {
                        if (!DrawRectEx(x1, y1, x2, y2)) return -1;
                        Reset();
                     }else
                        ++m_SearchPos;
                  }else
                     Reset();
               }
            }
            x = 0.0;
            for (i = 0; i < Count; i++)
            {
               if (!MarkSubString(ref x, m, Source[i])) return -1;
            }
            m_LastTextDir   = textDir;
            m_LastTextEndX1 = Width;
            m_LastTextEndY1 = 0.0;
            m_LastTextEndX4 = 0.0;
            m_LastTextEndY4 = m_GState.FontSize;
            Transform(m, ref m_LastTextEndX1, ref m_LastTextEndY1);
            Transform(m, ref m_LastTextEndX4, ref m_LastTextEndY4);
            return 0;
         }catch
         {
            return -1;
         }
      }

      public void MulMatrix(TCTM Matrix)
      {
         m_GState.Matrix = MulMatrix(m_GState.Matrix, Matrix);
      }

      private TCTM MulMatrix(TCTM M1, TCTM M2)
      {
         TCTM retval;
         retval.a = M2.a * M1.a + M2.b * M1.c;
         retval.b = M2.a * M1.b + M2.b * M1.d;
         retval.c = M2.c * M1.a + M2.d * M1.c;
         retval.d = M2.c * M1.b + M2.d * M1.d;
         retval.x = M2.x * M1.a + M2.y * M1.c + M1.x;
         retval.y = M2.x * M1.b + M2.y * M1.d + M1.y;
         return retval;
      }

      private void Reset()
      {
         m_HavePos   = false;
         m_SearchPos = 0;
      }

      public bool RestoreGState()
      {
         return m_Stack.Restore(ref m_GState);
      }

      public int SaveGState()
      {
         return m_Stack.Save(ref m_GState);
      }

      public void SetCharSpacing(double Value)
      {
         m_GState.CharSpacing = (float)Value;
      }

      public void SetFont(double FontSize, TFontType Type, IntPtr Font)
      {
         m_GState.ActiveFont = Font;
         m_GState.FontSize   = FontSize;
         m_GState.FontType   = Type;
         m_GState.SpaceWidth = (float)(m_PDF.GetSpaceWidth(Font, FontSize) * 0.5);
      }

      public void SetSearchText(String Text)
      {
         m_SearchText = Text;
         m_SearchPos = 0;
      }

      private void SetStartCoord(TCTM Matrix, double x)
      {
         m_x1 = x;
         m_y1 = 0.0;
         m_x4 = x;
         m_y4 = m_GState.FontSize;
         Transform(Matrix, ref m_x1, ref m_y1);
         Transform(Matrix, ref m_x4, ref m_y4);
         m_HavePos = true;
      }
      
      public void SetTextDrawMode(TDrawMode Mode)
      {
         m_GState.TextDrawMode = Mode;
      }

      public void SetTextScale(double Value)
      {
         m_GState.TextScale = (float)Value;
      }

      public void SetWordSpacing(double Value)
      {
         m_GState.WordSpacing = (float)Value;
      }

      private void Transform(TCTM M, ref double x, ref double y)
      {
         double tx = x;
         x = tx * M.a + y * M.c + M.x;
         y = tx * M.b + y * M.d + M.y;
      }
      protected enum TTextDir
      {
         tfLeftToRight    = 0,
         tfRightToLeft    = 1,
         tfTopToBottom    = 2,
         tfBottomToTop    = 4,
         tfNotInitialized = -1
      }

      protected TGState  m_GState;
      protected bool     m_HavePos;
      protected TTextDir m_LastTextDir;
      protected double   m_LastTextEndX1;
      protected double   m_LastTextEndY1;
      protected double   m_LastTextEndX4;
      protected double   m_LastTextEndY4;
      protected double   m_LastTextInfX;
      protected double   m_LastTextInfY;
      protected byte[]   m_OutBuf;
      protected int      m_OutLen;
      internal  CPDF     m_PDF;
      protected int      m_SearchPos;
      protected String   m_SearchText;
      protected int      m_SelCount;
      protected CStack   m_Stack;
      protected double   m_x1;
      protected double   m_x4;
      protected double   m_y1;
      protected double   m_y4;
   }
}
